"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Plus, TrendingUp, Eye, MousePointer, Wallet, BarChart3 } from "lucide-react"
import { useAuth } from "@/components/auth-provider"
import { useRouter } from "next/navigation"
import { Logo } from "@/components/logo"

interface UserAd {
  id: string
  title: string
  description: string
  imageUrl?: string
  targetUrl: string
  keywords: string[]
  dailyLimit: number
  monthlyLimit: number
  costPerClick: number
  costPerImpression: number
  totalSpent: number
  clicks: number
  impressions: number
  isActive: boolean
  createdAt: string
}

export default function AdsCenterPage() {
  const { user, logout, loading: authLoading } = useAuth()
  const router = useRouter()
  const [ads, setAds] = useState<UserAd[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (authLoading) return

    if (!user) {
      router.push("/ads-center/auth")
      return
    }
    fetchUserAds()
  }, [user, router, authLoading])

  const fetchUserAds = async () => {
    try {
      const response = await fetch(`/api/user-ads?userId=${user?.id}`)
      if (response.ok) {
        const data = await response.json()
        setAds(data.ads || [])
      }
    } catch (error) {
      console.error("Error fetching user ads:", error)
    } finally {
      setLoading(false)
    }
  }

  const handleEditAd = (adId: string) => {
    router.push(`/ads-center/edit/${adId}`)
  }

  const handleToggleAdStatus = async (adId: string, currentStatus: boolean) => {
    try {
      const response = await fetch(`/api/user-ads/${adId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ isActive: !currentStatus }),
      })

      if (response.ok) {
        setAds(ads.map((ad) => (ad.id === adId ? { ...ad, isActive: !currentStatus } : ad)))
      }
    } catch (error) {
      console.error("Error toggling ad status:", error)
    }
  }

  const handleDeleteAd = async (adId: string) => {
    if (!confirm("Are you sure you want to delete this ad?")) return

    try {
      const response = await fetch(`/api/user-ads/${adId}`, {
        method: "DELETE",
      })

      if (response.ok) {
        setAds(ads.filter((ad) => ad.id !== adId))
      }
    } catch (error) {
      console.error("Error deleting ad:", error)
    }
  }

  if (authLoading) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <p className="text-muted-foreground">Loading...</p>
      </div>
    )
  }

  if (!user) {
    return null
  }

  const totalSpent = ads.reduce((sum, ad) => sum + ad.totalSpent, 0)
  const totalClicks = ads.reduce((sum, ad) => sum + ad.clicks, 0)
  const totalImpressions = ads.reduce((sum, ad) => sum + ad.impressions, 0)
  const activeAds = ads.filter((ad) => ad.isActive).length

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b border-border bg-card">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <div className="flex items-center gap-4">
            <Logo size="sm" />
            <div>
              <h1 className="text-xl font-semibold text-foreground">Ads Center</h1>
              <p className="text-sm text-muted-foreground">Welcome back, {user.username}</p>
            </div>
          </div>
          <div className="flex items-center gap-4">
            <div className="text-right">
              <p className="text-sm text-muted-foreground">Wallet Balance</p>
              <p className="font-semibold text-primary">₹{user.walletBalance.toFixed(2)}</p>
            </div>
            <Button variant="outline" onClick={() => router.push("/ads-center/wallet")}>
              <Wallet className="w-4 h-4 mr-2" />
              Add Funds
            </Button>
            <Button variant="ghost" onClick={logout}>
              Logout
            </Button>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Active Ads</CardTitle>
              <TrendingUp className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-primary">{activeAds}</div>
              <p className="text-xs text-muted-foreground">of {ads.length} total ads</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Spent</CardTitle>
              <Wallet className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-primary">₹{totalSpent.toFixed(2)}</div>
              <p className="text-xs text-muted-foreground">this month</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Total Clicks</CardTitle>
              <MousePointer className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-primary">{totalClicks.toLocaleString()}</div>
              <p className="text-xs text-muted-foreground">all time</p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Impressions</CardTitle>
              <Eye className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-primary">{totalImpressions.toLocaleString()}</div>
              <p className="text-xs text-muted-foreground">all time</p>
            </CardContent>
          </Card>
        </div>

        {/* Ads Management */}
        <div className="flex justify-between items-center mb-6">
          <div>
            <h2 className="text-2xl font-semibold text-foreground">Your Advertisements</h2>
            <p className="text-muted-foreground">Create and manage your ad campaigns</p>
          </div>
          <Button onClick={() => router.push("/ads-center/create")} className="bg-primary hover:bg-primary/90">
            <Plus className="w-4 h-4 mr-2" />
            Create New Ad
          </Button>
        </div>

        {loading ? (
          <div className="text-center py-12">
            <p className="text-muted-foreground">Loading your ads...</p>
          </div>
        ) : ads.length === 0 ? (
          <Card>
            <CardContent className="text-center py-12">
              <BarChart3 className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-semibold text-foreground mb-2">No ads yet</h3>
              <p className="text-muted-foreground mb-4">Create your first advertisement to start reaching customers</p>
              <Button onClick={() => router.push("/ads-center/create")} className="bg-primary hover:bg-primary/90">
                <Plus className="w-4 h-4 mr-2" />
                Create Your First Ad
              </Button>
            </CardContent>
          </Card>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {ads.map((ad) => (
              <Card key={ad.id} className="hover:shadow-lg transition-shadow">
                <CardHeader>
                  <div className="flex justify-between items-start">
                    <CardTitle className="text-lg">{ad.title}</CardTitle>
                    <Badge variant={ad.isActive ? "default" : "secondary"}>{ad.isActive ? "Active" : "Paused"}</Badge>
                  </div>
                  <CardDescription className="line-clamp-2">{ad.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  {ad.imageUrl && (
                    <img
                      src={ad.imageUrl || "/placeholder.svg"}
                      alt={ad.title}
                      className="w-full h-32 object-cover rounded-md mb-4"
                    />
                  )}
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">Clicks:</span>
                      <span className="font-medium">{ad.clicks}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">Impressions:</span>
                      <span className="font-medium">{ad.impressions}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-muted-foreground">Spent:</span>
                      <span className="font-medium text-primary">₹{ad.totalSpent.toFixed(2)}</span>
                    </div>
                  </div>
                  <div className="flex gap-2 mt-4">
                    <Button
                      variant="outline"
                      size="sm"
                      className="flex-1 bg-transparent"
                      onClick={() => handleEditAd(ad.id)}
                    >
                      Edit
                    </Button>
                    <Button
                      variant="outline"
                      size="sm"
                      className="flex-1 bg-transparent"
                      onClick={() => handleToggleAdStatus(ad.id, ad.isActive)}
                    >
                      {ad.isActive ? "Pause" : "Resume"}
                    </Button>
                    {/* Added delete button */}
                    <Button
                      variant="outline"
                      size="sm"
                      className="flex-1 bg-transparent"
                      onClick={() => handleDeleteAd(ad.id)}
                    >
                      Delete
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
